import os
import string
from math import *

class DeformAmif:

  def __init__(self, shape_file):

    self.relaxation = 0.1
    self.timestep   = 0.01
    self.mass       = 1.0
    self.length     = 1.0

    self.surf_result_ami = "(none)"
    self.shape_file = shape_file


  #-----------------------------------------------------------------------------
  # read the latest update from the TAU parameter file
  #-----------------------------------------------------------------------------
  def read_para_file(self, para_file):

    f = open(para_file, 'r')
    a = f.readlines()

    for line in a:
      words = string.split(line)
      if (len(words) == 4 and 
          words[0] == 'Surface' and
          words[1] == 'output' and
          words[2] == 'filename:'):
        self.surf_result_ami = words[3] + ".ami"

    f.close()


  #-----------------------------------------------------------------------------
  # read the ASCII output created by 'tau2plt' for the AMIF format output
  # the file contains the point global-ids and the surface forces
  #-----------------------------------------------------------------------------
  def read_surf_file(self):

    f = open(self.surf_result_ami, 'r')
    a = f.readlines()

    self.gid = []

    self.coords0 = {}
    self.forces = {}

    i = 0
    for line in a:
      words = string.split(line)
      if (len(words) == 6 and words[0] == 'AGRID'):
        self.gid.append(int(words[1]))
        x0 = float(words[2])
        y0 = float(words[3])
        z0 = float(words[4])
        self.coords0[self.gid[i]] = [x0, y0, z0]
        i = i + 1
      if (len(words) == 5 and words[0] == 'AGFORCE'):
        l_id = int(words[1])
        fx = float(words[2])
        fy = float(words[3])
        fz = float(words[4])
        self.forces[l_id] = [fx, fy, fz]

    f.close()


  #-----------------------------------------------------------------------------
  # deform the surface; simple function using Fz as input
  #-----------------------------------------------------------------------------
  def calc_deformation(self):

    dt = self.timestep
    m  = self.mass
    l  = self.length
    relax = self.relaxation

    Fz = 0.0
    for i in self.gid:
      Fz = Fz + self.forces[i][2]

    print "\nSummation of Fz: %20.12f\n" %(Fz)

    self.coords1 = {}

    for i in self.gid:
      x = self.coords0[i][0]
      y = self.coords0[i][1]
      z = self.coords0[i][2] + relax * (Fz * abs(y)/l)/m * dt * dt

      self.coords1[i] = [x, y, z]


  #-----------------------------------------------------------------------------
  # write ASCII deformation file using the AMIF format
  #-----------------------------------------------------------------------------
  def write_deformation(self, para_file, step):

    key = "AGDEFO"

    #-------------- Open AMIF file for writing ------------
    self.new_filename = "%s.step_%d.ami" %(self.shape_file, step)
    amif_file = open(self.new_filename, 'w')

    for i in self.gid:
      x = self.coords1[i][0]
      y = self.coords1[i][1]
      z = self.coords1[i][2]
      line = "%s %d %16.9e %16.9e %16.9e\n" %(key, i, x, y, z)
      amif_file.write(line)

    amif_file.close()


  #-----------------------------------------------------------------------------
  #
  #-----------------------------------------------------------------------------
  def create_deformed_surface(self, para_file, step):

    self.read_para_file(para_file)
    self.read_surf_file()
    self.calc_deformation()
    self.write_deformation(para_file, step)

    return self.new_filename


